// Gem 5 Sample.cpp
//

#pragma comment( lib, "lua5.1.lib" )
#define WIN32_LEAN_AND_MEAN
#define VC_EXTRALEAN

#include <iostream>
#include <windows.h>
#include <assert.h>
#include <list>
#include <conio.h>

extern "C" {
	#include "lualib.h"
	#include "lauxlib.h"
}

#include "lua.h"

///////////////////////////////////////////////////////////////////////////////
// LuaObject and derivatives
///////////////////////////////////////////////////////////////////////////////

//An example game object with both C++ and Lua logic.
class LuaObject
{
public:
	LuaObject( ) { }

	//Bind the lua object to a self table at index
	void BindToScript( lua_State* L, int idx )
	{
		//Push the index onto the stack
		lua_pushvalue( L, idx );						//STACK:	self, ...

		//Store the __c_pointer in the script table
		lua_pushstring( L, "__c_pointer" );				//STACK:	"__c_pointer", self, ...
		lua_pushlightuserdata( L, this );				//STACK:	lo, "__c_pointer", self, ...
		lua_settable( L, -3 );							//STACK:	self, ...
			
		//Map the engine side pointer to the script side table through the lua registry.
		lua_pushlightuserdata( L, this );				//STACK:	lo, self, ...
		lua_pushvalue( L, -2 );							//STACK:	self, lo, self, ...
		lua_settable( L, LUA_REGISTRYINDEX );			//STACK:	self, ...
		lua_pop( L, -1 );								//STACK:	...
	}
};

//Called in lua to create a new lua object and bind the script/c++ sides
static int NewLuaObject( lua_State *L )
{
	LuaObject* lo = new LuaObject( );
	
	//Arg 1 is the "self" table from script.
	lo->BindToScript( L, 1 );			//STACK:	self

	return 0;
}

class EventManager : public LuaObject
{
public:
	void FireEvent( lua_State* L, std::string eventName )
	{
		//Call the lua FireEvent on this event manager object.
		lua_pushlightuserdata( L, this );			//STACK:	em
		lua_gettable( L, LUA_REGISTRYINDEX );		//STACK:	self
		lua_pushstring( L, "FireEvent" );			//STACK:	"FireEvent", self
		lua_gettable( L, -2 );						//STACK:	FireFunc, self
		lua_pushvalue( L, -2 );						//STACK:	self, FireFunc, self
		lua_pushstring( L, eventName.c_str() );		//STACK:	self, eventName, FireFunc, self
		lua_pcall( L, 2, 0, 0 );					//STACK:	self
	}

	void Update( lua_State* L )
	{
		//Call the lua function Update on this event manager object.
		lua_pushlightuserdata( L, this );			//STACK:	em
		lua_gettable( L, LUA_REGISTRYINDEX );		//STACK:	self
		lua_pushstring( L, "Update" );				//STACK:	"Update", self
		lua_gettable( L, -2 );						//STACK:	UpdateFunc, self
		lua_pushvalue( L, -2 );						//STACK:	self, UpdateFunc, self
		lua_pcall( L, 1, 0, 0 );					//STACK:	self
	}
};

static EventManager* sEM = NULL;

//Called in lua to create the event manager singleton
static int NewEventManager( lua_State *L )
{
	if ( sEM )
	{
		return 0;
	}

	sEM = new EventManager( );
	
	//Arg 1 is the "self" table from script.
	sEM->BindToScript( L, 1 );			//STACK:	self

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Main Program
///////////////////////////////////////////////////////////////////////////////

std::string GetPath()
{
	char pathBuffer[ MAX_PATH ];
	GetModuleFileName( NULL, pathBuffer, MAX_PATH );	
	std::string path( pathBuffer );
	return path.substr( 0, path.find_last_of( "\\" ) + 1 ).append( "..\\" );
}

int _main( int argc, char* argv[] )
{
	lua_State* L = lua_open();
	luaopen_base( L );
	luaopen_table( L );
	luaopen_string( L );
	luaopen_math( L );
	luaopen_debug( L );

	lua_register( L, "NewLuaObject", NewLuaObject );
	lua_register( L, "NewEventManager", NewEventManager );

	std::string path = GetPath();
	int error = luaL_loadfile( L, path.append( "Gem 5 Sample\\_main.lua" ).c_str() );

	if ( !error )
	{
		error = lua_pcall( L, 0, LUA_MULTRET, 0 );
	}

	if ( error )
	{
		std::cerr << lua_tostring( L, -1 ) << std::endl;
		lua_pop( L, 1 );
	}

	int loopCount = 0;
	bool bContinue = true;
	while ( bContinue )
	{
		sEM->Update( L );

		++loopCount;
		if ( loopCount >= 50 ) 
		{
			loopCount = 0;
			sEM->FireEvent( L, "OnUpdateScriptObjects" );
		}

		if ( _kbhit() )
		{
			int keyCode = _getch();
			switch ( keyCode )
			{
			case 27: //escape
				{
					bContinue = false;
					break;
				}
			case 75: //left arrow
				{
					sEM->FireEvent( L, "OnArrowKeyLeft" );
					break;
				}
			case 72: //up arrow
				{
					sEM->FireEvent( L, "OnArrowKeyUp" );
					break;
				}
			case 77: //right arrow
				{
					sEM->FireEvent( L, "OnArrowKeyRight" );
					break;
				}
			case 80: //down arrow
				{
					sEM->FireEvent( L, "OnArrowKeyDown" );
					break;
				}		
			}
		}
		Sleep( 100 );
	}

	lua_close( L );

	system( "pause" );
	return 0;
}

